// ====================================
// NEW: SIMPLE THREE.JS PARTICLE FIELD BACKGROUND
// ====================================

function initParticleField() {
    if (typeof window.THREE === 'undefined') {
        console.error('THREE.js library not found. Particle field failed to load.');
        return;
    }

    const container = document.getElementById('webgl-background-container');
    if (!container) return;

    const width = container.clientWidth;
    const height = container.clientHeight;

    const scene = new THREE.Scene();
    const camera = new THREE.PerspectiveCamera(75, width / height, 0.1, 1000);
    camera.position.z = 5;

    const renderer = new THREE.WebGLRenderer({ antialias: true, alpha: true });
    renderer.setSize(width, height);
    renderer.setPixelRatio(window.devicePixelRatio);
    renderer.domElement.style.position = 'absolute';
    renderer.domElement.style.top = '0';
    renderer.domElement.style.left = '0';
    renderer.domElement.style.width = '100%';
    renderer.domElement.style.height = '100%';
    container.appendChild(renderer.domElement);

    const particleCount = 500; // Reduced particle count
    const geometry = new THREE.BufferGeometry();
    const positions = new Float32Array(particleCount * 3);
    const colors = new Float32Array(particleCount * 3);
    const tempColor = new THREE.Color();

    for (let i = 0; i < particleCount; i++) {
        positions[i * 3 + 0] = (Math.random() - 0.5) * 50;
        positions[i * 3 + 1] = (Math.random() - 0.5) * 50;
        positions[i * 3 + 2] = (Math.random() - 0.5) * 50;

        if (i % 3 === 0) {
            tempColor.set('#ffffff');
        } else if (i % 3 === 1) {
            tempColor.set('#5227FF');
        } else {
            tempColor.set('#f59e0b');
        }
        colors[i * 3 + 0] = tempColor.r;
        colors[i * 3 + 1] = tempColor.g;
        colors[i * 3 + 2] = tempColor.b;
    }

    geometry.setAttribute('position', new THREE.BufferAttribute(positions, 3));
    geometry.setAttribute('color', new THREE.BufferAttribute(colors, 3));

    const material = new THREE.PointsMaterial({
        size: 0.05,
        vertexColors: true,
        transparent: true,
        blending: THREE.AdditiveBlending,
        sizeAttenuation: true
    });

    const points = new THREE.Points(geometry, material);
    scene.add(points);

    const animate = () => {
        requestAnimationFrame(animate);
        points.rotation.x += 0.0005;
        points.rotation.y += 0.0008;
        const time = Date.now() * 0.00005;
        camera.position.x = Math.cos(time) * 10;
        camera.position.y = Math.sin(time) * 10;
        camera.lookAt(scene.position);
        renderer.render(scene, camera);
    };

    const onResize = () => {
        const newWidth = container.clientWidth;
        const newHeight = container.clientHeight;
        camera.aspect = newWidth / newHeight;
        camera.updateProjectionMatrix();
        renderer.setSize(newWidth, newHeight);
    };

    window.addEventListener('resize', onResize);
    animate();
}

// ====================================
// CORE SITE LOGIC
// ====================================

const ownerBios = {
    'AKIGAI': {
        name: 'AKIGAI (Aki)',
        description: "The visionary leader and technical backbone of Sugarrush SMP. Aki focuses on server development, performance optimization, and charting the future direction of the community.",
        details: "<strong>Role:</strong> Owner, Server Lead, Founder, Lead Developer, Server Architect<br><strong>Favorite Server Feature:</strong> Farming cows [Sorry GAI]",
        image: 'https://i.ibb.co/7JJnmhRL/photo-output-3.jpg',
        discord: 'https://discord.gg/ptAMCCq8d5',
        youtube: 'https://www.youtube.com/@AKIGAIgamer'
    },
    'ISHU': {
        name: 'ISHU',
        description: "The creative heart of Sugarrush SMP. Ishu manages the community, organizes events, and designs the aesthetic, custom world map to make every corner feel unique.",
        details: "<strong>Role:</strong> Owner, Community Manager, Creative Director<br><strong>Joined:</strong> Early 2023<br><strong>Favorite Server Feature:</strong> The Seasonal Events",
        image: 'https://i.ibb.co/Ng9G1Ltj/content.png',
        discord: 'https://discord.gg/5977SPpDwr',
        youtube: 'https://www.youtube.com/@Real_ishu568'
    }
};

const teamBios = {
    'BLAZ': {
        name: 'BLAZ',
        image: 'https://i.ibb.co/7JJnmhRL/photo-output-3.jpg',
        details: `
            <p><strong>Roles at Server:</strong> Developer, Technical Support</p>
            <p><strong>In-Game Name:</strong> Blaz_X</p>
            <p><strong>In-Game Role:</strong> Builder</p>
            <p><strong>Discord ID:</strong> blaz#1234</p>
            <p><strong>Time Available:</strong> Evenings (IST)</p>
        `
    },
    'LEXY': {
        name: 'LEXY',
        image: 'https://i.ibb.co/7JJnmhRL/photo-output-3.jpg',
        details: `
            <p><strong>Roles at Server:</strong>Sr.ADMIN</p>
            <p><strong>In-Game Name:</strong> LexyPlayzZ_</p>
            <p><strong>In-Game Role:</strong> Jr.DEV, Sr.Admin</p>
            <p><strong>Discord ID:</strong> lexyji_</p>
            <p><strong>Time Available:</strong>10am - 7pm</p>
        `
    },
    'RANBIR': {
        name: 'RANBIR',
        image: 'https://i.ibb.co/7JJnmhRL/photo-output-3.jpg',
        details: `
            <p><strong>Roles at Server:</strong> Admin</p>
            <p><strong>In-Game Name:</strong> S_Ranbir</p>
            <p><strong>In-Game Role:</strong> Admin,Moderator</p>
            <p><strong>Discord ID:</strong> sr_altruist</p>
            <p><strong>Time Available:</strong> 6pm - 6am</p>
        `
    }
};

function showNotification(message) {
    const notification = document.getElementById('notification');
    if (!notification) return;
    notification.textContent = message;
    notification.classList.add('show');
    setTimeout(() => {
        notification.classList.remove('show');
    }, 3000);
}

function copyToClipboard(textToCopy, extraInfo = '') {
    const fullText = extraInfo ? `${textToCopy}\n${extraInfo}` : textToCopy;
    const textArea = document.createElement("textarea");
    textArea.value = fullText;
    textArea.style.position = "fixed";
    textArea.style.opacity = "0";
    document.body.appendChild(textArea);
    textArea.focus();
    textArea.select();
    try {
        document.execCommand('copy');
        showNotification(`Copied: ${textToCopy}`);
    } catch (err) {
        showNotification('Failed to copy. Please copy manually.');
    }
    document.body.removeChild(textArea);
}

function showModal(modal) {
    if (modal) modal.classList.remove('hidden');
}

function hideModal(modal) {
    if (modal) modal.classList.add('hidden');
}

function createFallingFlowers(container) {
    if (!container) return;
    container.innerHTML = '';
    
    const flowerTypes = ['🌸', '🌺', '💮', '🌷'];
    for (let i = 0; i < 15; i++) {
        const flower = document.createElement('div');
        flower.className = 'flower';
        flower.textContent = flowerTypes[Math.floor(Math.random() * flowerTypes.length)];
        flower.style.left = `${Math.random() * 100}%`;
        flower.style.fontSize = `${1 + Math.random() * 1.5}rem`;
        flower.style.opacity = `${0.4 + Math.random() * 0.5}`;
        flower.style.animationDuration = `${7 + Math.random() * 8}s`;
        flower.style.animationDelay = `${Math.random() * 5}s`;
        container.appendChild(flower);
    }
}

function createTechEmojis(container) {
    if (!container) return;
    container.innerHTML = '';
    
    const techEmojiTypes = ['🤖', '👾'];
    for (let i = 0; i < 15; i++) {
        const emoji = document.createElement('div');
        emoji.className = 'flower'; // Re-use the same class for styling
        emoji.textContent = techEmojiTypes[Math.floor(Math.random() * techEmojiTypes.length)];
        emoji.style.left = `${Math.random() * 100}%`;
        emoji.style.fontSize = `${1 + Math.random() * 1.5}rem`;
        emoji.style.opacity = `${0.4 + Math.random() * 0.5}`;
        emoji.style.animationDuration = `${7 + Math.random() * 8}s`;
        emoji.style.animationDelay = `${Math.random() * 5}s`;
        container.appendChild(emoji);
    }
}

function showBioModal(ownerKey) {
    const bio = ownerBios[ownerKey];
    const modal = document.getElementById('bioModal');
    const modalContent = document.getElementById('bioModalContent');
    if (!bio || !modal || !modalContent) return;
    
    document.getElementById('bioName').textContent = bio.name;
    document.getElementById('bioDescription').textContent = bio.description;
    document.getElementById('bioDetails').innerHTML = bio.details;
    document.getElementById('bioImage').src = bio.image;
    document.getElementById('bioImage').alt = `${bio.name} Profile Picture`;
    document.getElementById('bioDiscordLink').href = bio.discord;
    document.getElementById('bioYoutubeLink').href = bio.youtube;
    
    modalContent.classList.add('purple-theme');
    const flowerContainer = document.createElement('div');
    flowerContainer.className = 'falling-flowers';
    modalContent.prepend(flowerContainer);
    createFallingFlowers(flowerContainer);

    showModal(modal);
}

function showTeamModal(memberKey) {
    const bio = teamBios[memberKey];
    const modal = document.getElementById('teamModal');
    const modalContent = document.getElementById('teamModalContent');
    if (!bio || !modal || !modalContent) return;

    document.getElementById('teamMemberName').textContent = bio.name;
    document.getElementById('teamMemberDetails').innerHTML = bio.details;

    modalContent.classList.add('purple-theme');
    
    const existingFlowers = modalContent.querySelector('.falling-flowers');
    if(existingFlowers) existingFlowers.remove();
    const flowerContainer = document.createElement('div');
    flowerContainer.className = 'falling-flowers';
    modalContent.prepend(flowerContainer);
    createTechEmojis(flowerContainer);

    showModal(modal);
}


function initScrollAnimations() {
    const observer = new IntersectionObserver((entries) => {
        entries.forEach(entry => {
            if (entry.isIntersecting) {
                const delay = entry.target.dataset.delay || 0;
                setTimeout(() => {
                    entry.target.classList.add('visible');
                }, delay * 1000);
                observer.unobserve(entry.target);
            }
        });
    }, { threshold: 0.1, rootMargin: '0px 0px -50px 0px' });
    document.querySelectorAll('.fade-in-up, .slide-in-left, .slide-in-right, .scale-in').forEach(el => observer.observe(el));
}

function createMinecraftParticles() {
    const container = document.getElementById('minecraftParticles');
    if (!container) return;
    const texts = ['⛏️', '💎', '🔥', '⚔️', '🛡️', '🍎', '🍖', '🍞', '🏆', '🌟'];
    for (let i = 0; i < 15; i++) {
        const particle = document.createElement('div');
        particle.className = 'particle';
        particle.textContent = texts[Math.floor(Math.random() * texts.length)];
        particle.style.left = `${Math.random() * 100}%`;
        particle.style.animationDelay = `${Math.random() * 10}s`;
        particle.style.animationDuration = `${10 + Math.random() * 15}s`;
        container.appendChild(particle);
    }
}

function initFullWidthNav() {
    const nav = document.getElementById('full-width-nav');
    if (!nav || typeof gsap === 'undefined') {
        if (typeof gsap === 'undefined') console.error("GSAP is not loaded.");
        return;
    }

    const pills = Array.from(nav.querySelectorAll('.pill'));
    const mobileMenuButton = document.getElementById('mobile-menu-button');
    const mobileMenuPopover = document.getElementById('mobile-menu-popover');
    let isMobileMenuOpen = false;

    pills.forEach((pill) => {
        const circle = document.createElement('span');
        circle.className = 'hover-circle';
        pill.prepend(circle);

        const label = pill.textContent.trim();
        pill.innerHTML = `<span class="label-stack"><span class="pill-label">${label}</span><span class="pill-label-hover">${label}</span></span>`;
        pill.prepend(circle);

        let tl;

        const layout = () => {
            const rect = pill.getBoundingClientRect();
            const { width: w, height: h } = rect;
            const R = ((w * w) / 4 + h * h) / (2 * h);
            const D = Math.ceil(2 * R) + 2;
            const delta = Math.ceil(R - Math.sqrt(Math.max(0, R * R - (w * w) / 4))) + 1;
            const originY = D - delta;

            circle.style.width = `${D}px`;
            circle.style.height = `${D}px`;
            circle.style.bottom = `-${delta}px`;

            gsap.set(circle, { xPercent: -50, scale: 0, transformOrigin: `50% ${originY}px` });
            const labelEl = pill.querySelector('.pill-label');
            const hoverLabelEl = pill.querySelector('.pill-label-hover');
            
            gsap.set(labelEl, { y: 0 });
            gsap.set(hoverLabelEl, { y: h + 12, opacity: 0 });

            tl?.kill();
            tl = gsap.timeline({ paused: true });
            const ease = 'power3.easeOut';
            tl.to(circle, { scale: 1.2, duration: 0.4, ease })
              .to(labelEl, { y: -(h + 8), duration: 0.4, ease }, 0)
              .to(hoverLabelEl, { y: 0, opacity: 1, duration: 0.4, ease }, 0);
        };

        layout();
        window.addEventListener('resize', layout);

        pill.addEventListener('mouseenter', () => tl.timeScale(1).play());
        pill.addEventListener('mouseleave', () => tl.timeScale(1.5).reverse());
    });

    mobileMenuButton.addEventListener('click', () => {
        isMobileMenuOpen = !isMobileMenuOpen;
        const lines = mobileMenuButton.querySelectorAll('.hamburger-line');
        const ease = 'power3.easeOut';

        if (isMobileMenuOpen) {
            gsap.to(lines[0], { rotation: 45, y: 3, duration: 0.3, ease });
            gsap.to(lines[1], { rotation: -45, y: -3, duration: 0.3, ease });
            gsap.set(mobileMenuPopover, { visibility: 'visible' });
            gsap.fromTo(mobileMenuPopover, { opacity: 0, y: 10 }, { opacity: 1, y: 0, duration: 0.3, ease });
        } else {
            gsap.to(lines[0], { rotation: 0, y: 0, duration: 0.3, ease });
            gsap.to(lines[1], { rotation: 0, y: 0, duration: 0.3, ease });
            gsap.to(mobileMenuPopover, {
                opacity: 0, y: 10, duration: 0.2, ease,
                onComplete: () => gsap.set(mobileMenuPopover, { visibility: 'hidden' })
            });
        }
    });
    
    mobileMenuPopover.querySelectorAll('a').forEach(link => {
       link.addEventListener('click', () => {
           if(isMobileMenuOpen) mobileMenuButton.click();
       }) 
    });
}

function initContinuousShuffle(elementId) {
    const element = document.getElementById(elementId);
    if (!element || typeof gsap === 'undefined') return;

    const originalText = element.textContent;
    const chars = "ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789";
    element.innerHTML = ''; 

    originalText.split('').forEach(char => {
        const wrapper = document.createElement('span');
        wrapper.className = 'shuffle-char-wrapper';
        
        const inner = document.createElement('span');
        inner.className = 'shuffle-char-inner';
        
        const originalCharSpan = document.createElement('span');
        originalCharSpan.className = 'shuffle-char';
        originalCharSpan.textContent = char;
        inner.appendChild(originalCharSpan);

        for (let i = 0; i < 10; i++) {
            const randomChar = document.createElement('span');
            randomChar.className = 'shuffle-char';
            randomChar.textContent = chars[Math.floor(Math.random() * chars.length)];
            inner.appendChild(randomChar);
        }
        
        wrapper.appendChild(inner);
        element.appendChild(wrapper);
    });

    const charInners = gsap.utils.toArray("#" + elementId + " .shuffle-char-inner");
    
    const shuffleRandomChar = () => {
        const randomInner = gsap.utils.random(charInners);
        const charHeight = randomInner.querySelector('.shuffle-char').offsetHeight;
        
        gsap.timeline()
           .to(randomInner, { y: `-${charHeight * gsap.utils.random(1, 10, 1)}px`, duration: 0.1, ease: 'power2.in' })
           .to(randomInner, { y: 0, duration: 1.5, ease: 'elastic.out(1, 0.3)' }, '+=0.1');
    };

    setInterval(shuffleRandomChar, 2000);
}

function initVariableProximityEffect(elementId) {
    const element = document.getElementById(elementId);
    if (!element || typeof gsap === 'undefined') {
        return;
    }
    const originalText = element.innerText;
    element.innerHTML = ''; // Clear original text
    originalText.split('').forEach(char => {
        const span = document.createElement('span');
        // Use non-breaking space for actual spaces to maintain layout
        span.innerText = char === ' ' ? '\u00A0' : char;
        element.appendChild(span);
    });

    const spans = Array.from(element.children);
    const maxWeight = 900;
    const minWeight = 300;
    const proximityThreshold = 150; // Pixels

    element.addEventListener('mousemove', (e) => {
        const { left } = element.getBoundingClientRect();
        const mouseX = e.clientX - left;

        spans.forEach(span => {
            const { left: spanLeft, width: spanWidth } = span.getBoundingClientRect();
            // Calculate center of the character relative to the parent element
            const spanCenter = (spanLeft - left) + spanWidth / 2;
            const distance = Math.abs(mouseX - spanCenter);

            let weight;
            if (distance < proximityThreshold) {
                // Map distance to weight. Closer = heavier.
                const proximityFactor = 1 - (distance / proximityThreshold);
                weight = minWeight + (maxWeight - minWeight) * proximityFactor;
            } else {
                weight = minWeight;
            }
            
            // Use GSAP for smooth animation
            gsap.to(span, {
                fontVariationSettings: `'wght' ${weight}`,
                duration: 0.3,
                ease: 'power2.out'
            });
        });
    });

    element.addEventListener('mouseleave', () => {
        // Animate all spans back to the minimum weight
        gsap.to(spans, {
            fontVariationSettings: `'wght' ${minWeight}`,
            duration: 0.5,
            ease: 'power2.out'
        });
    });
}

function initRotatingTextAnimation(containerSelector) {
    const container = document.querySelector(containerSelector);
    if (!container) return;

    const words = container.querySelectorAll('.rotating-text-word');
    if (words.length < 2) return;

    let currentIndex = 0;
    words[currentIndex].classList.add('active'); // Set initial active word

    setInterval(() => {
        const currentWord = words[currentIndex];
        const nextIndex = (currentIndex + 1) % words.length;
        const nextWord = words[nextIndex];

        currentWord.classList.remove('active');
        nextWord.classList.add('active');

        currentIndex = nextIndex;
    }, 2000); // Cycle every 2 seconds
}

// ====================================
// Loading Screen Typing Effect
// ====================================
function initTypingEffect(elementId, text, speed = 200) {
    const element = document.getElementById(elementId);
    if (!element) return;
    let i = 0;
    element.innerHTML = ""; // Clear text initially

    function typeWriter() {
        if (i < text.length) {
            element.innerHTML += text.charAt(i);
            i++;
            setTimeout(typeWriter, speed);
        }
    }
    typeWriter();
}

// ====================================
// HYPERSPEED INTRO LOGIC
// ====================================

function initHyperspeedIntro() {
    const container = document.getElementById('lights');
    const loader = document.getElementById('loader');
    if (!container || !loader) {
        console.error('Loader or lights container not found.');
        if(loader) loader.classList.add('hidden');
        initScrollAnimations();
        return;
    }

    // Start the typing effect after a short delay
    setTimeout(() => {
        initTypingEffect('loading-text', 'HELLO!!!!', 200);
    }, 500);

    // Check for libraries
    if (typeof THREE === 'undefined' || typeof POSTPROCESSING === 'undefined' || typeof gsap === 'undefined') {
        console.error('THREE.js, Postprocessing, or GSAP is not loaded.');
        loader.classList.add('hidden');
        initScrollAnimations();
        return;
    }
    
    // Destructure necessary classes from the global POSTPROCESSING object
    const { BloomEffect, EffectComposer, EffectPass, RenderPass, SMAAEffect, SMAAPreset } = POSTPROCESSING;
    
    // --- All the Hyperspeed classes and shaders go here --- //
    
    const hyperspeedPresets = {
        one: {
            distortion: 'turbulentDistortion',
            length: 400,
            roadWidth: 10,
            islandWidth: 2,
            lanesPerRoad: 3,
            fov: 90,
            fovSpeedUp: 150,
            speedUp: 2,
            carLightsFade: 0.4,
            totalSideLightSticks: 20,
            lightPairsPerRoadWay: 40,
            shoulderLinesWidthPercentage: 0.05,
            brokenLinesWidthPercentage: 0.1,
            brokenLinesLengthPercentage: 0.5,
            lightStickWidth: [0.12, 0.5],
            lightStickHeight: [1.3, 1.7],
            movingAwaySpeed: [60, 80],
            movingCloserSpeed: [-120, -160],
            carLightsLength: [400 * 0.03, 400 * 0.2],
            carLightsRadius: [0.05, 0.14],
            carWidthPercentage: [0.3, 0.5],
            carShiftX: [-0.8, 0.8],
            carFloorSeparation: [0, 5],
            colors: {
                roadColor: 0x080808,
                islandColor: 0x0a0a0a,
                background: 0x000000,
                shoulderLines: 0x131318,
                brokenLines: 0x131318,
                leftCars: [0xd856bf, 0x6750a2, 0xc247ac],
                rightCars: [0x03b3c3, 0x0e5ea5, 0x324555],
                sticks: 0x03b3c3
            }
        }
    };
    
    const turbulentUniforms = {
        uFreq: { value: new THREE.Vector4(4, 8, 8, 1) },
        uAmp: { value: new THREE.Vector4(25, 5, 10, 10) }
    };
    
    let nsin = val => Math.sin(val) * 0.5 + 0.5;

    const distortions = {
        turbulentDistortion: {
            uniforms: turbulentUniforms,
            getDistortion: `
                uniform vec4 uFreq;
                uniform vec4 uAmp;
                float nsin(float val){
                    return sin(val) * 0.5 + 0.5;
                }
                #define PI 3.14159265358979
                float getDistortionX(float progress){
                    return (
                        cos(PI * progress * uFreq.r + uTime) * uAmp.r +
                        pow(cos(PI * progress * uFreq.g + uTime * (uFreq.g / uFreq.r)), 2. ) * uAmp.g
                    );
                }
                float getDistortionY(float progress){
                    return (
                        -nsin(PI * progress * uFreq.b + uTime) * uAmp.b +
                        -pow(nsin(PI * progress * uFreq.a + uTime / (uFreq.b / uFreq.a)), 5.) * uAmp.a
                    );
                }
                vec3 getDistortion(float progress){
                    return vec3(
                        getDistortionX(progress) - getDistortionX(0.0125),
                        getDistortionY(progress) - getDistortionY(0.0125),
                        0.
                    );
                }
            `,
            getJS: (progress, time) => {
                const uFreq = turbulentUniforms.uFreq.value;
                const uAmp = turbulentUniforms.uAmp.value;
                const getX = p => Math.cos(Math.PI * p * uFreq.x + time) * uAmp.x + Math.pow(Math.cos(Math.PI * p * uFreq.y + time * (uFreq.y / uFreq.x)), 2) * uAmp.y;
                const getY = p => -nsin(Math.PI * p * uFreq.z + time) * uAmp.z - Math.pow(nsin(Math.PI * p * uFreq.w + time / (uFreq.z / uFreq.w)), 5) * uAmp.w;
                let distortion = new THREE.Vector3(getX(progress) - getX(progress + 0.007), getY(progress) - getY(progress + 0.007), 0);
                let lookAtAmp = new THREE.Vector3(-2, -5, 0);
                let lookAtOffset = new THREE.Vector3(0, 0, -10);
                return distortion.multiply(lookAtAmp).add(lookAtOffset);
            }
        }
    };
    
    class App {
        constructor(container, options = {}) {
            this.options = options;
            this.options.distortion = distortions[options.distortion];
            
            this.container = container;
            this.renderer = new THREE.WebGLRenderer({ antialias: false, alpha: true });
            this.renderer.setSize(container.offsetWidth, container.offsetHeight, false);
            this.renderer.setPixelRatio(window.devicePixelRatio);
            this.composer = new EffectComposer(this.renderer);
            container.append(this.renderer.domElement);
            
            this.camera = new THREE.PerspectiveCamera(options.fov, container.offsetWidth / container.offsetHeight, 0.1, 10000);
            this.camera.position.z = -5;
            this.camera.position.y = 8;
            this.scene = new THREE.Scene();
            
            let fog = new THREE.Fog(options.colors.background, options.length * 0.2, options.length * 500);
            this.scene.fog = fog;
            this.fogUniforms = {
                fogColor: { value: fog.color },
                fogNear: { value: fog.near },
                fogFar: { value: fog.far }
            };
            
            this.clock = new THREE.Clock();
            this.disposed = false;
            
            this.road = new Road(this, options);
            this.leftCarLights = new CarLights(this, options, options.colors.leftCars, options.movingAwaySpeed, new THREE.Vector2(0, 1 - options.carLightsFade));
            this.rightCarLights = new CarLights(this, options, options.colors.rightCars, options.movingCloserSpeed, new THREE.Vector2(1, 0 + options.carLightsFade));
            this.leftSticks = new LightsSticks(this, options);
            
            this.fovTarget = options.fov;
            this.speedUpTarget = 0;
            this.speedUp = 0;
            this.timeOffset = 0;
            
            this.tick = this.tick.bind(this);
            this.init = this.init.bind(this);
            this.onWindowResize = this.onWindowResize.bind(this);
            window.addEventListener('resize', this.onWindowResize);
        }
        
        onWindowResize() {
            const width = this.container.offsetWidth;
            const height = this.container.offsetHeight;
            this.camera.aspect = width / height;
            this.camera.updateProjectionMatrix();
            this.renderer.setSize(width, height);
            this.composer.setSize(width, height);
        }
        
        initPasses() {
            this.renderPass = new RenderPass(this.scene, this.camera);
            this.bloomPass = new EffectPass(this.camera, new BloomEffect({ luminanceThreshold: 0.2, luminanceSmoothing: 0, resolutionScale: 1 }));
            const smaaPass = new EffectPass(this.camera, new SMAAEffect({ preset: SMAAPreset.MEDIUM }));
            
            this.renderPass.renderToScreen = false;
            this.bloomPass.renderToScreen = false;
            smaaPass.renderToScreen = true;
            
            this.composer.addPass(this.renderPass);
            this.composer.addPass(this.bloomPass);
            this.composer.addPass(smaaPass);
        }
        
        init() {
            this.initPasses();
            const options = this.options;
            this.road.init();
            this.leftCarLights.init();
            this.leftCarLights.mesh.position.setX(-options.roadWidth / 2 - options.islandWidth / 2);
            this.rightCarLights.init();
            this.rightCarLights.mesh.position.setX(options.roadWidth / 2 + options.islandWidth / 2);
            this.leftSticks.init();
            this.leftSticks.mesh.position.setX(-(options.roadWidth + options.islandWidth / 2));
            this.tick();
        }
        
        update(delta) {
            let lerpPercentage = Math.exp(-(-60 * Math.log2(1 - 0.1)) * delta);
            this.speedUp += lerp(this.speedUp, this.speedUpTarget, lerpPercentage, 0.00001);
            this.timeOffset += this.speedUp * delta;
            
            let time = this.clock.elapsedTime + this.timeOffset;
            
            this.rightCarLights.update(time);
            this.leftCarLights.update(time);
            this.leftSticks.update(time);
            this.road.update(time);
            
            let updateCamera = false;
            let fovChange = lerp(this.camera.fov, this.fovTarget, lerpPercentage);
            if (fovChange !== 0) {
                this.camera.fov += fovChange * delta * 6;
                updateCamera = true;
            }
            
            if (this.options.distortion.getJS) {
                const distortion = this.options.distortion.getJS(0.025, time);
                this.camera.lookAt(new THREE.Vector3(this.camera.position.x + distortion.x, this.camera.position.y + distortion.y, this.camera.position.z + distortion.z));
                updateCamera = true;
            }
            
            if (updateCamera) {
                this.camera.updateProjectionMatrix();
            }
        }
        
        render(delta) {
            this.composer.render(delta);
        }
        
        dispose() {
            this.disposed = true;
            window.removeEventListener('resize', this.onWindowResize);
            if (this.renderer) this.renderer.dispose();
            if (this.composer) this.composer.dispose();
            if (this.scene) this.scene.clear();
        }
        
        tick() {
            if (this.disposed) return;
            const delta = this.clock.getDelta();
            this.render(delta);
            this.update(delta);
            requestAnimationFrame(this.tick);
        }
    }

    const random = base => {
        if (Array.isArray(base)) return Math.random() * (base[1] - base[0]) + base[0];
        return Math.random() * base;
    };
    
    const pickRandom = arr => {
        if (Array.isArray(arr)) return arr[Math.floor(Math.random() * arr.length)];
        return arr;
    };

    function lerp(current, target, speed = 0.1, limit = 0.001) {
        let change = (target - current) * speed;
        if (Math.abs(change) < limit) {
            change = target - current;
        }
        return change;
    }

    const carLightsFragment = `
        #define USE_FOG;
        varying vec3 vColor;
        varying vec2 vUv; 
        uniform vec2 uFade;
        void main() {
            vec3 color = vec3(vColor);
            float alpha = smoothstep(uFade.x, uFade.y, vUv.x);
            gl_FragColor = vec4(color, alpha);
            if (gl_FragColor.a < 0.0001) discard;
        }`;

    const carLightsVertex = `
        #define USE_FOG;
        attribute vec3 aOffset;
        attribute vec3 aMetrics;
        attribute vec3 aColor;
        uniform float uTravelLength;
        uniform float uTime;
        varying vec2 vUv; 
        varying vec3 vColor; 
        #include <getDistortion_vertex>
        void main() {
            vec3 transformed = position.xyz;
            float radius = aMetrics.r;
            float myLength = aMetrics.g;
            float speed = aMetrics.b;
            transformed.xy *= radius;
            transformed.z *= myLength;
            transformed.z += myLength - mod(uTime * speed + aOffset.z, uTravelLength);
            transformed.xy += aOffset.xy;
            float progress = abs(transformed.z / uTravelLength);
            transformed.xyz += getDistortion(progress);
            vec4 mvPosition = modelViewMatrix * vec4(transformed, 1.);
            gl_Position = projectionMatrix * mvPosition;
            vUv = uv;
            vColor = aColor;
        }`;

    class CarLights {
        constructor(webgl, options, colors, speed, fade) {
            this.webgl = webgl;
            this.options = options;
            this.colors = colors;
            this.speed = speed;
            this.fade = fade;
        }
        init() {
            const options = this.options;
            let curve = new THREE.LineCurve3(new THREE.Vector3(0, 0, 0), new THREE.Vector3(0, 0, -1));
            let geometry = new THREE.TubeGeometry(curve, 40, 1, 8, false);
            let instanced = new THREE.InstancedBufferGeometry().copy(geometry);
            instanced.instanceCount = options.lightPairsPerRoadWay * 2;
            let laneWidth = options.roadWidth / options.lanesPerRoad;
            let aOffset = [], aMetrics = [], aColor = [];
            let colors = Array.isArray(this.colors) ? this.colors.map(c => new THREE.Color(c)) : new THREE.Color(this.colors);

            for (let i = 0; i < options.lightPairsPerRoadWay; i++) {
                let radius = random(options.carLightsRadius), length = random(options.carLightsLength), speed = random(this.speed);
                let carLane = i % options.lanesPerRoad;
                let laneX = carLane * laneWidth - options.roadWidth / 2 + laneWidth / 2;
                let carWidth = random(options.carWidthPercentage) * laneWidth;
                let carShiftX = random(options.carShiftX) * laneWidth;
                laneX += carShiftX;
                let offsetY = random(options.carFloorSeparation) + radius * 1.3, offsetZ = -random(options.length);
                
                aOffset.push(laneX - carWidth / 2, offsetY, offsetZ, laneX + carWidth / 2, offsetY, offsetZ);
                aMetrics.push(radius, length, speed, radius, length, speed);
                let color = pickRandom(colors);
                aColor.push(color.r, color.g, color.b, color.r, color.g, color.b);
            }
            instanced.setAttribute('aOffset', new THREE.InstancedBufferAttribute(new Float32Array(aOffset), 3, false));
            instanced.setAttribute('aMetrics', new THREE.InstancedBufferAttribute(new Float32Array(aMetrics), 3, false));
            instanced.setAttribute('aColor', new THREE.InstancedBufferAttribute(new Float32Array(aColor), 3, false));
            
            let material = new THREE.ShaderMaterial({
                fragmentShader: carLightsFragment, vertexShader: carLightsVertex, transparent: true,
                uniforms: Object.assign({ uTime: { value: 0 }, uTravelLength: { value: options.length }, uFade: { value: this.fade } }, this.webgl.fogUniforms, options.distortion.uniforms)
            });
            material.onBeforeCompile = shader => { shader.vertexShader = shader.vertexShader.replace('#include <getDistortion_vertex>', options.distortion.getDistortion); };
            
            this.mesh = new THREE.Mesh(instanced, material);
            this.mesh.frustumCulled = false;
            this.webgl.scene.add(this.mesh);
        }
        update(time) { this.mesh.material.uniforms.uTime.value = time; }
    }
    
    // ... Other classes (LightsSticks, Road) and shaders can be pasted here in the same manner
     const sideSticksVertex = `
        #define USE_FOG;
        attribute float aOffset;
        attribute vec3 aColor;
        attribute vec2 aMetrics;
        uniform float uTravelLength;
        uniform float uTime;
        varying vec3 vColor;
        mat4 rotationY( in float angle ) {
            return mat4( cos(angle),0,sin(angle),0, 0,1.0,0,0, -sin(angle),0,cos(angle),0, 0,0,0,1);
        }
        #include <getDistortion_vertex>
        void main(){
            vec3 transformed = position.xyz;
            float width = aMetrics.x;
            float height = aMetrics.y;
            transformed.xy *= vec2(width, height);
            float time = mod(uTime * 60. * 2. + aOffset, uTravelLength);
            transformed = (rotationY(3.14/2.) * vec4(transformed,1.)).xyz;
            transformed.z += - uTravelLength + time;
            float progress = abs(transformed.z / uTravelLength);
            transformed.xyz += getDistortion(progress);
            transformed.y += height / 2.;
            transformed.x += -width / 2.;
            vec4 mvPosition = modelViewMatrix * vec4(transformed, 1.);
            gl_Position = projectionMatrix * mvPosition;
            vColor = aColor;
        }`;

    const sideSticksFragment = `
        #define USE_FOG;
        varying vec3 vColor;
        void main(){
            vec3 color = vec3(vColor);
            gl_FragColor = vec4(color,1.);
        }`;
        
    class LightsSticks {
        constructor(webgl, options) { this.webgl = webgl; this.options = options; }
        init() {
            const options = this.options;
            const geometry = new THREE.PlaneGeometry(1, 1);
            let instanced = new THREE.InstancedBufferGeometry().copy(geometry);
            let totalSticks = options.totalSideLightSticks;
            instanced.instanceCount = totalSticks;
            let stickoffset = options.length / (totalSticks - 1);
            const aOffset = [], aColor = [], aMetrics = [];
            let colors = Array.isArray(options.colors.sticks) ? options.colors.sticks.map(c => new THREE.Color(c)) : new THREE.Color(options.colors.sticks);
            for (let i = 0; i < totalSticks; i++) {
                let width = random(options.lightStickWidth), height = random(options.lightStickHeight);
                aOffset.push((i - 1) * stickoffset * 2 + stickoffset * Math.random());
                let color = pickRandom(colors);
                aColor.push(color.r, color.g, color.b);
                aMetrics.push(width, height);
            }
            instanced.setAttribute('aOffset', new THREE.InstancedBufferAttribute(new Float32Array(aOffset), 1, false));
            instanced.setAttribute('aColor', new THREE.InstancedBufferAttribute(new Float32Array(aColor), 3, false));
            instanced.setAttribute('aMetrics', new THREE.InstancedBufferAttribute(new Float32Array(aMetrics), 2, false));
            const material = new THREE.ShaderMaterial({
                fragmentShader: sideSticksFragment, vertexShader: sideSticksVertex, side: THREE.DoubleSide,
                uniforms: Object.assign({ uTravelLength: { value: options.length }, uTime: { value: 0 } }, this.webgl.fogUniforms, options.distortion.uniforms)
            });
            material.onBeforeCompile = shader => { shader.vertexShader = shader.vertexShader.replace('#include <getDistortion_vertex>', options.distortion.getDistortion); };
            this.mesh = new THREE.Mesh(instanced, material);
            this.mesh.frustumCulled = false;
            this.webgl.scene.add(this.mesh);
        }
        update(time) { this.mesh.material.uniforms.uTime.value = time; }
    }

    const roadVertex = `
        #define USE_FOG;
        uniform float uTime;
        varying vec2 vUv;
        uniform float uTravelLength;
        #include <getDistortion_vertex>
        void main() {
            vec3 transformed = position.xyz;
            vec3 distortion = getDistortion((transformed.y + uTravelLength / 2.) / uTravelLength);
            transformed.x += distortion.x;
            transformed.z += distortion.y;
            transformed.y += -1. * distortion.z;
            vec4 mvPosition = modelViewMatrix * vec4(transformed, 1.);
            gl_Position = projectionMatrix * mvPosition;
            vUv = uv;
        }`;

    const roadFragment = `
        #define USE_FOG;
        varying vec2 vUv;
        uniform vec3 uColor;
        uniform float uTime;
        uniform float uLanes;
        uniform vec3 uBrokenLinesColor;
        uniform vec3 uShoulderLinesColor;
        uniform float uShoulderLinesWidthPercentage;
        uniform float uBrokenLinesWidthPercentage;
        uniform float uBrokenLinesLengthPercentage;
        void main() {
            vec2 uv = vUv;
            vec3 color = vec3(uColor);
            uv.y = mod(uv.y + uTime * 0.05, 1.);
            float laneWidth = 1.0 / uLanes;
            float brokenLineWidth = laneWidth * uBrokenLinesWidthPercentage;
            float shoulderLinesWidth = 1.0 * uShoulderLinesWidthPercentage;
            
            float brokenLines = step(1. - brokenLineWidth, fract(uv.x * uLanes)) * step(0.4, fract(uv.y * 10.));
            color = mix(color, uBrokenLinesColor, brokenLines);

            float shoulderLines = step(1. - shoulderLinesWidth, uv.x) + step(uv.x, shoulderLinesWidth);
            color = mix(color, uShoulderLinesColor, shoulderLines);
            
            gl_FragColor = vec4(color, 1.);
        }`;

    const islandFragment = `
        #define USE_FOG;
        varying vec2 vUv;
        uniform vec3 uColor;
        void main() {
            vec3 color = vec3(uColor);
            gl_FragColor = vec4(color, 1.);
        }`;
        
    class Road {
        constructor(webgl, options) { this.webgl = webgl; this.options = options; this.uTime = { value: 0 }; }
        createPlane(side, width, isRoad) {
            const options = this.options;
            const geometry = new THREE.PlaneGeometry(isRoad ? options.roadWidth : options.islandWidth, options.length, 20, 100);
            let uniforms = { uTravelLength: { value: options.length }, uColor: { value: new THREE.Color(isRoad ? options.colors.roadColor : options.colors.islandColor) }, uTime: this.uTime };
            if (isRoad) {
                uniforms = Object.assign(uniforms, {
                    uLanes: { value: options.lanesPerRoad },
                    uBrokenLinesColor: { value: new THREE.Color(options.colors.brokenLines) },
                    uShoulderLinesColor: { value: new THREE.Color(options.colors.shoulderLines) },
                    uShoulderLinesWidthPercentage: { value: options.shoulderLinesWidthPercentage },
                    uBrokenLinesLengthPercentage: { value: options.brokenLinesLengthPercentage },
                    uBrokenLinesWidthPercentage: { value: options.brokenLinesWidthPercentage }
                });
            }
            const material = new THREE.ShaderMaterial({
                fragmentShader: isRoad ? roadFragment : islandFragment, vertexShader: roadVertex, side: THREE.DoubleSide,
                uniforms: Object.assign(uniforms, this.webgl.fogUniforms, options.distortion.uniforms)
            });
            material.onBeforeCompile = shader => { shader.vertexShader = shader.vertexShader.replace('#include <getDistortion_vertex>', options.distortion.getDistortion); };
            const mesh = new THREE.Mesh(geometry, material);
            mesh.rotation.x = -Math.PI / 2;
            mesh.position.z = -options.length / 2;
            mesh.position.x += (this.options.islandWidth / 2 + options.roadWidth / 2) * side;
            this.webgl.scene.add(mesh);
            return mesh;
        }
        init() {
            this.leftRoadWay = this.createPlane(-1, this.options.roadWidth, true);
            this.rightRoadWay = this.createPlane(1, this.options.roadWidth, true);
            this.island = this.createPlane(0, this.options.islandWidth, false);
        }
        update(time) { this.uTime.value = time; }
    }
    
    // --- App instantiation and animation --- //
    
    const effectOptions = hyperspeedPresets.one;
    
    const myApp = new App(container, effectOptions);
    myApp.init();

    const tl = gsap.timeline({
        onComplete: () => {
            initScrollAnimations(); // Start other animations after intro
            if (myApp) myApp.dispose(); // Clean up Three.js scene
        }
    });

    tl.to(myApp, {
        speedUpTarget: myApp.options.speedUp,
        duration: 2,
        ease: 'power2.in'
    }, 0)
    .to(myApp.camera, {
        fov: myApp.options.fovSpeedUp,
        duration: 2,
        ease: 'power2.in'
    }, 0)
    .to(loader, {
        delay: 2.5, // Hold the fast effect for a bit
        duration: 1,
        opacity: 0,
        ease: 'power2.inOut',
        onComplete: () => {
            loader.classList.add('hidden');
        }
    });
}


document.addEventListener('DOMContentLoaded', () => {
    
    initHyperspeedIntro();
    initParticleField();
    
    const joinModal = document.getElementById('joinModal');
    const bioModal = document.getElementById('bioModal');
    const teamModal = document.getElementById('teamModal');
    const joinNowBtn = document.getElementById('joinNowBtn');
    const closeJoinModalBtn = document.getElementById('closeJoinModalBtn');
    const closeBioModalBtn = document.getElementById('closeBioModalBtn');
    const closeTeamModalBtn = document.getElementById('closeTeamModalBtn');
    const activePlayersEl = document.getElementById('active-players');
    const totalMembersEl = document.getElementById('total-members');
    const scrollToTopBtn = document.getElementById('scrollToTopBtn');
    
    initFullWidthNav();
    initContinuousShuffle('shuffle-logo');
    initVariableProximityEffect('owners-title-effect');
    initDecryptionEffect('decrypted-title');
    initRotatingTextAnimation('.rotating-text-container');


    window.addEventListener('scroll', () => {
        if (window.scrollY > 50) {
            scrollToTopBtn.classList.add('show');
        } else {
            scrollToTopBtn.classList.remove('show');
        }
    });

    if (joinNowBtn && joinModal) joinNowBtn.addEventListener('click', () => showModal(joinModal));
    if (closeJoinModalBtn && joinModal) closeJoinModalBtn.addEventListener('click', () => hideModal(joinModal));
    if (closeBioModalBtn && bioModal) closeBioModalBtn.addEventListener('click', () => hideModal(bioModal));
    if (closeTeamModalBtn && teamModal) closeTeamModalBtn.addEventListener('click', () => hideModal(teamModal));
    
    [joinModal, bioModal, teamModal].forEach(modal => {
        if(modal) {
            modal.addEventListener('click', (e) => {
                if (e.target === modal) {
                    hideModal(modal);
                }
            });
        }
    });

    document.body.addEventListener('keydown', (e) => {
        if (e.key === 'Enter' && document.activeElement && typeof document.activeElement.onclick === 'function') {
            document.activeElement.onclick(e);
        }
        if (e.key === 'Escape') {
            hideModal(joinModal);
            hideModal(bioModal);
            hideModal(teamModal);
        }
    });

    function animateCountUp(el) {
        if(!el) return;
        const target = parseInt(el.textContent.replace(/,/g, ''), 10);
        if (isNaN(target)) return;
        
        let current = 0;
        const duration = 2000;
        const steps = 100;
        const stepTime = duration / steps;
        const increment = target / steps;

        const timer = setInterval(() => {
            current += increment;
            if (current >= target) {
                clearInterval(timer);
                el.textContent = target.toLocaleString();
            } else {
                el.textContent = Math.ceil(current).toLocaleString();
            }
        }, stepTime);
    }
    animateCountUp(activePlayersEl);
    animateCountUp(totalMembersEl);

    const yearSpan = document.getElementById('current-year');
    if (yearSpan) {
        yearSpan.textContent = new Date().getFullYear();
    }

    function initCircularText() {
        const textContainer = document.getElementById('circular-text');
        if (!textContainer) return;
        const text = "• MADE BY. RANBIR SR •"; 
        text.split('').forEach((char, i) => {
            const span = document.createElement('span');
            span.innerText = char;
            const rotate = (360 / text.length) * i;
            span.style.transform = `rotate(${rotate}deg)`;
            textContainer.appendChild(span);
        });
    }
    initCircularText();
    
    function initDecryptionEffect(elementId) {
        const target = document.getElementById(elementId);
        if (!target) return;
        const letters = "ABCDEFGHIJKLMNOPQRSTUVWXYZ!@#$%^&*()";
        const originalValue = target.dataset.value;
        let interval = null;

        target.onmouseover = () => {  
            let iteration = 0;
            clearInterval(interval);
            interval = setInterval(() => {
                target.innerText = target.innerText.split("").map((letter, index) => {
                    if(index < iteration) return originalValue[index];
                    return letters[Math.floor(Math.random() * letters.length)]
                }).join("");
                if(iteration >= originalValue.length) clearInterval(interval);
                iteration += 1 / 3;
            }, 30);
        }
    }
    
    function initStoryScroll() {
        const steps = document.querySelectorAll('.story-step');
        const images = document.querySelectorAll('.story-image');
        if (!steps.length || !images.length) return;

        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const stepIndex = parseInt(entry.target.dataset.step);
                    images.forEach((img, imgIndex) => {
                        img.classList.toggle('active', imgIndex === stepIndex);
                    });
                }
            });
        }, { threshold: 0.5 });

        steps.forEach(step => observer.observe(step));
    }
    initStoryScroll();

    const heroBg = document.querySelector('.hero-bg');
    if (heroBg) {
        window.addEventListener('scroll', () => {
            const scrollY = window.pageYOffset;
            heroBg.style.transform = `translate3d(0, ${scrollY * 0.4}px, 0)`;
        });
    }
    
    feather.replace();
    createMinecraftParticles();
});

